/*
 *  Juggle version 1.0, Copyright (C) 1995 G. Hutchings
 *  Juggle comes with ABSOLUTELY NO WARRANTY.
 *  This is free software, and you are welcome to redistribute it
 *  under certain conditions; see the file COPYING for details.
 */

/* Interface to juggling display functions */

#include "config.h"

/* Functions you need to define in the display package. */

/*
 * This sets up the graphic display.  Called before any of the
 * functions below.
 */
void juggle_init();

/*
 * This sets up the next picture frame.  Typically just clears the
 * balls from the old picture.
 */
void juggle_frame();

/*
 * This draws a ball on the display.  Args are: ball number (starting
 * from 0), time it was thrown, time it will be caught, and the x/y
 * position in a unit box.  This is where to put all your flashy
 * effects (coloured 3D-looking knives spinning in mid-air, etc. 8-).
 */
void juggle_ball(int num, int throw, int catch, double x, double y);

/*
 * This updates the display once all the balls have been drawn.
 */
void juggle_update();

/*
 * This returns whether the user has had enough.  Could also be used
 * to carry out user commands (e.g. change the frame speed).
 */
int juggle_quit();

/*
 * This closes down the graphic display.
 */
void juggle_end();

/* Available functions that you don't need to define. */

/*
 * This sets the juggle timestep, if not already set by the -s option.
 * Call this from juggle_init() if your display package needs to use a
 * different default timestep.  Note: step is silently truncated to the
 * interval (0, 1).  If the force flag is set, the timestep is reset
 * even if already set by -s.
 */
void juggle_set_step(double t, int force);

/* External variables.  These are all set before juggle_init() is */
/* called.  Unless stated otherwise, it's best to treat these as */
/* read-only. */

/*
 * Name by which the program was invoked.
 */
extern char *juggle_prog;

/*
 * String representation of the current juggling pattern.
 */
extern char *juggle_pattern;

/*
 * Descriptive name of the juggling pattern (if any), set by the -n
 * option.  NULL means pattern has no name.
 */
extern char *juggle_name;

/*
 * No. of balls in the current pattern.  Useful for initialising ball
 * attributes, etc.
 */
extern int juggle_nballs;

/*
 * Max. thrown `height' of a juggling ball.  Might be useful for scaling
 * the display.
 */
extern int juggle_maxht;

/*
 * Whether throws of height 2 are being held in hand.  Throws of height
 * 2 represent the smallest throw from one hand to itself.  In practice,
 * jugglers usually just hold the ball instead of throwing it.  This
 * effect is enabled with the -h option.
 */
extern int juggle_hold;

/*
 * Current time.  One time `unit' is the time it takes for the smallest
 * throw (one hand to the other at the lowest height).  Can't think why
 * this would be needed by display routines, but here it is anyway.
 */
extern double juggle_time;

/*
 * Current timestep.  Don't set this manually - use juggle_set_step()
 * above.
 */
extern double juggle_step;

/*
 * Time it takes to do one `cycle' of the pattern.
 */
extern int juggle_cycle;

/*
 * Debugging flag.
 */
extern int juggle_debug;

/*
 * Return whether an (x, y) ball position represents a held ball.
 * Might be useful, for instance, if you're putting spin effects in
 * so you can stop spinning temporarily.
 */
#define HELD(x, y)	(ABS(y) < 0.01)

/*
 * Test which hand a ball is coming from, or going to.  You can use
 * these macros on the 'throw' and 'catch' arguments to juggle_ball().
 */
#define LEFT_HAND(n)	((n) % 2 == 0)
#define RIGHT_HAND(n)	((n) % 2 != 0)

/* Other useful macros. */
#define ABS(x)		((x) > 0 ? (x) : -(x))
#define MIN(a, b)	((a) < (b) ? (a) : (b))
#define MAX(a, b)	((a) > (b) ? (a) : (b))
#define SWAP(i, j)	{ int tmp; tmp = i; i = j; j = tmp; }
