/*
 *  Juggle version 1.0, Copyright (C) 1995 G. Hutchings
 *  Juggle comes with ABSOLUTELY NO WARRANTY.
 *  This is free software, and you are welcome to redistribute it
 *  under certain conditions; see the file COPYING for details.
 */

/* The main program */

#ifndef lint
static char rcsid[] = "$Id: juggle.c,v 1.7 1995/03/30 15:46:14 g_hutchi Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include "juggle.h"

/* Program version */
#define VERSION		"1.0"

/* Default timestep */
#define DEFAULT_STEP	0.2

/* The main drawing effect */
#define X(t)		(sin(M_PI*(t)/2))
#define Y(t)		(4*(t)*(1-(t)))

/* Throw height from a pattern position */
#define THROW(n)	pattern[(n) % period]

/* External variables (see juggle.h) */
char *juggle_prog;
char *juggle_pattern = NULL;
char *juggle_name = NULL;
int juggle_nballs = 0;
int juggle_maxht = 0;
int juggle_cycle = 0;
int juggle_hold = 0;
int juggle_debug = 0;
double juggle_time;
double juggle_step = -1.0;

/* Forward declarations */
void error(char *msg);
void usage();

int
main(int argc, char *argv[])
{
    int c, i, j, period = 6, *pattern, *initial, *balls;
    extern char *optarg;
    extern int opterr;
    double x, y, t;

    /* What's my name? */
    juggle_prog = argv[0];

    /* Parse arguments */
    opterr = 0;
    while ((c = getopt(argc, argv, "hn:p:r:s:vD")) != EOF) {
	switch (c) {
	case 'h':
	    juggle_hold = 1;
	    break;
	case 'n':
	    juggle_name = optarg;
	    break;
	case 'p':
	    juggle_pattern = optarg;
	    period = strlen(juggle_pattern);
	    for (i = 0; juggle_pattern[i] != '\0'; i++) {
		if (!isdigit(juggle_pattern[i]))
		    error("invalid pattern string");
	    }
	    break;
	case 'r':
	    period = atoi(optarg);
	    if (period < 2 || period > 9)
		error("invalid period");
	    break;
	case 's':
	    juggle_set_step(atof(optarg), 1);
	    break;
	case 'v':
	    printf("Juggle v%s, (c) Glenn Hutchings 1995\n", VERSION);
	    exit(0);
	case 'D':
	    juggle_debug = 1;
	    break;
	default:
	    usage();
	}
    }

    /* Initialise random stream */
    RANDINIT;

    /* Set up pattern */
    pattern = ALLOC(int, period);

    if (juggle_pattern != NULL) {
	/* User-specified pattern */
	int *permute = ALLOC(int, period);

	/* Build pattern */
	for (i = 0; i < period; i++) {
	    pattern[i] = juggle_pattern[i] - '0';
	    permute[i] = (pattern[i] + i) % period;
	}

	/* Check it's valid */
	for (i = 0; i < period; i++) {
	    for (j = 0; j < i; j++) {
		if (permute[i] == permute[j])
		    error("pattern not jugglable");
	    }
	}

	DEALLOC(permute);
    } else {
	/* Random pattern */
	int nswaps = 100;

	/* Create random permutation */
	for (i = 0; i < period; i++)
	    pattern[i] = i;

	while (nswaps-- > 0) {
	    i = RANDOM(period);
	    j = RANDOM(period);
	    SWAP(pattern[i], pattern[j]);
	}

	/* Convert it to pattern */
	juggle_pattern = ALLOC(char, period);

	for (i = 0; i < period; i++) {
	    pattern[i] -= i;
	    while (pattern[i] < 0)
		pattern[i] += period;
	    if (pattern[i] == 0 && RANDOM(2))
		pattern[i] += period;
	    juggle_pattern[i] = '0' + pattern[i];
	}

	juggle_pattern[period] = '\0';
    }

    /* Find maximum throwing height and pattern length */
    for (i = 0; i < period; i++) {
	juggle_maxht = MAX(juggle_maxht, pattern[i]);
	juggle_cycle += pattern[i];
    }

    /* Initialise balls */
    juggle_nballs = juggle_cycle / period;
    balls = ALLOC(int, juggle_nballs);

    /* Find initial ball positions and start time */
    initial = ALLOC(int, juggle_cycle);
    for (i = 0; i < juggle_cycle; i++)
	initial[i] = 0;

    for (i = 0; i < juggle_nballs; i++) {
	for (j = 0; initial[j] != 0 || THROW(j) == 0; j++);
	while (j < juggle_cycle) {
	    initial[j] = 1;
	    balls[i] = j;
	    j += THROW(j);
	}
    }

    DEALLOC(initial);
    juggle_time = juggle_cycle;

    /* Initialise display */
    juggle_init();

    /* Set default timestep */
    juggle_set_step(DEFAULT_STEP, 0);

    /* Juggle them balls! */
    while (!juggle_quit()) {
	/* Set up next frame */
	juggle_frame();

	/* Draw balls on it */
	for (i = 0; i < juggle_nballs; i++) {
	    int throw, catch;

	    /* Find throw and catch times */
	    while (juggle_time > balls[i] + THROW(balls[i]))
		balls[i] += THROW(balls[i]);
	    throw = balls[i];
	    catch = throw + THROW(throw);
	    t = (juggle_time - throw) / (catch - throw);

	    /* Calculate x/y position in unit box */
	    if (LEFT_HAND(throw))
		x = (LEFT_HAND(catch) ? 0.0 : X(t));
	    else
		x = (LEFT_HAND(catch) ? 1 - X(t) : 1.0);

	    if (juggle_hold && THROW(throw) == 2)
		y = 0.0;
	    else
		y = Y(t) * THROW(throw) / juggle_maxht;

	    /* Put a ball there */
	    juggle_ball(i, throw, catch, x, y);
	}

	/* Update display */
	juggle_update();

	/* Next timestep */
	juggle_time += juggle_step;
    }

    /* Tidy up and exit */
    juggle_end();
    return 0;
}

/* Set the juggle timestep if not already done */
void
juggle_set_step(double t, int force)
{
    if (juggle_step < 0.0 || force)
	juggle_step = t;

    /* Silently enforce limits */
    juggle_step = MIN(juggle_step, 1.0);
    juggle_step = MAX(juggle_step, 0.001);
}

/* Print an error message and exit */
void
error(char *msg)
{
    fprintf(stderr, "%s: Error: %s\n", juggle_prog, msg);
    exit(1);
}

/* Print a usage message and exit */
void
usage()
{
    fprintf(stderr, "Usage: %s", juggle_prog);
    fprintf(stderr, " [-h]");
    fprintf(stderr, " [-n <name>]");
    fprintf(stderr, " [-p <pattern>]");
    fprintf(stderr, " [-r <period>]");
    fprintf(stderr, " [-s <step>]");
    fprintf(stderr, " [-v]");
    fprintf(stderr, "\n");
    exit(1);
}
